<?php
/**
 * Dynamic css
 *
 * @package xts
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

use XTS\Options;
use XTS\Presets;

/**
 * Dynamic css class
 *
 * @since 1.0.0
 */
class WOODMART_Themesettingscss {
	public $storage;

	/**
	 * Set up all properties
	 */
	public function __construct() {
		add_action( 'xts_before_theme_settings', array( $this, 'reset_data' ), 100 );
		add_action( 'xts_before_theme_settings', array( $this, 'write_file' ), 200 );
		add_action( 'wp', array( $this, 'print_styles' ), 300 );

		$this->storage = new WOODMART_Stylesstorage( $this->get_file_name() );
	}

	/**
	 * Get all css.
	 *
	 * @since 1.0.0
	 */
	private function get_all_css() {
		$css  = Options::get_instance()->get_css_output();
		$css .= $this->get_icons_font_css();
		$css .= $this->get_theme_settings_css();
		$css .= $this->get_custom_fonts_css();
		$css .= $this->get_custom_css();

		return $css;
	}

	/**
	 * Get file name.
	 *
	 * @since 1.0.0
	 */
	private function get_file_name() {
		$active_presets = Presets::get_active_presets();
		$preset_id      = isset( $active_presets[0] ) ? $active_presets[0] : 'default';
		return 'theme_settings_' . $preset_id;
	}

	/**
	 * Write file.
	 *
	 * @since 1.0.0
	 */
	public function reset_data() {
		if ( ! isset( $_GET['settings-updated'] ) ) {
			return;
		}

		$this->storage->reset_data();
	}

	/**
	 * Write file.
	 *
	 * @since 1.0.0
	 */
	public function write_file() {
		if ( ! isset( $_GET['page'] ) || ( isset( $_GET['page'] ) && 'xtemos_options' !== $_GET['page'] ) ) {
			return;
		}

		$this->storage->write( $this->get_all_css() );
	}

	/**
	 * Print styles.
	 *
	 * @since 1.0.0
	 */
	public function print_styles() {
		if ( ! $this->storage->is_css_exists() ) {
			$this->storage->write( $this->get_all_css() );
		}

		$this->storage->print_styles();
	}

	/**
	 * Get custom css.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_custom_css() {
		$output          = '';
		$custom_css      = woodmart_get_opt( 'custom_css' );
		$css_desktop     = woodmart_get_opt( 'css_desktop' );
		$css_tablet      = woodmart_get_opt( 'css_tablet' );
		$css_wide_mobile = woodmart_get_opt( 'css_wide_mobile' );
		$css_mobile      = woodmart_get_opt( 'css_mobile' );

		if ( $custom_css ) {
			$output .= $custom_css;
		}

		if ( $css_desktop ) {
			$output .= '@media (min-width: 1025px) { ' . $css_desktop . ' }';
		}

		if ( $css_tablet ) {
			$output .= '@media (min-width: 768px) and (max-width: 1024px) {' . $css_tablet . ' }';
		}

		if ( $css_wide_mobile ) {
			$output .= '@media (min-width: 577px) and (max-width: 767px) { ' . $css_wide_mobile . ' }';
		}

		if ( $css_mobile ) {
			$output .= '@media (max-width: 576px) { ' . $css_mobile . ' }';
		}

		return $output;
	}

	/**
	 * Get custom fonts css.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_custom_fonts_css() {
		$fonts = woodmart_get_opt( 'multi_custom_fonts' );

		$output       = '';
		$font_display = woodmart_get_opt( 'google_font_display' );

		if ( isset( $fonts['{{index}}'] ) ) {
			unset( $fonts['{{index}}'] );
		}

		if ( ! $fonts ) {
			return $output;
		}

		foreach ( $fonts as $key => $value ) {
			$eot   = $this->get_custom_font_url( $value['font-eot'] );
			$woff  = $this->get_custom_font_url( $value['font-woff'] );
			$woff2 = $this->get_custom_font_url( $value['font-woff2'] );
			$ttf   = $this->get_custom_font_url( $value['font-ttf'] );
			$svg   = $this->get_custom_font_url( $value['font-svg'] );

			if ( ! $value['font-name'] ) {
				continue;
			}

			$output .= '@font-face {';
			$output .= 'font-family: "' . sanitize_text_field( $value['font-name'] ) . '";';

			if ( $eot ) {
				$output .= 'src: url("' . esc_url( $eot ) . '");';
				$output .= 'src: url("' . esc_url( $eot ) . '#iefix") format("embedded-opentype"),';
			}

			if ( $woff ) {
				$output .= 'url("' . esc_url( $woff ) . '") format("woff"),';
			}

			if ( $woff2 ) {
				$output .= 'url("' . esc_url( $woff2 ) . '") format("woff2"),';
			}

			if ( $ttf ) {
				$output .= 'url("' . esc_url( $ttf ) . '") format("truetype"),';
			}

			if ( $svg ) {
				$output .= 'url("' . esc_url( $svg ) . '#' . sanitize_text_field( $value['font-name'] ) . '") format("svg");';
			}

			if ( $value['font-weight'] ) {
				$output .= 'font-weight: ' . sanitize_text_field( $value['font-weight'] ) . ';';
			} else {
				$output .= 'font-weight: normal;';
			}

			if ( 'disable' !== $font_display ) {
				$output .= 'font-display:' . $font_display . ';';
			}

			$output .= 'font-style: normal;';
			$output .= '}';
		}

		return $output;
	}

	/**
	 * Icons font css.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_icons_font_css() {
		$output = '';
		$url    = woodmart_remove_https( WOODMART_THEME_DIR );

		$font_display = woodmart_get_opt( 'icons_font_display' );

		// Our font.
		$output .= '@font-face {
			font-weight: normal;
			font-style: normal;
			font-family: "woodmart-font";
			src: url("' . $url . '/fonts/woodmart-font.eot");
			src: url("' . $url . '/fonts/woodmart-font.eot?#iefix") format("embedded-opentype"),
			url("' . $url . '/fonts/woodmart-font.woff") format("woff"),
			url("' . $url . '/fonts/woodmart-font.woff2") format("woff2"),
			url("' . $url . '/fonts/woodmart-font.ttf") format("truetype"),
			url("' . $url . '/fonts/woodmart-font.svg#woodmart-font") format("svg");';

		if ( 'disable' !== $font_display ) {
			$output .= 'font-display:' . $font_display . ';';
		}

		$output .= '}';

		if ( ! woodmart_get_opt( 'disable_font_awesome_theme_css' ) ) {
			$prefix = '';
			if ( woodmart_get_opt( 'light_font_awesome_version' ) ) {
				$prefix = '-light';
			}

			// FontAwesome
			$output .= '@font-face {
				font-family: "FontAwesome";
				src: url("' . $url . '/fonts/fontawesome-webfont' . $prefix . '.eot?v=4.7.0");
				src: url("' . $url . '/fonts/fontawesome-webfont' . $prefix . '.eot?#iefix&v=4.7.0") format("embedded-opentype"),
				url("' . $url . '/fonts/fontawesome-webfont' . $prefix . '.woff2?v=4.7.0") format("woff2"),
				url("' . $url . '/fonts/fontawesome-webfont' . $prefix . '.woff?v=4.7.0") format("woff"),
				url("' . $url . '/fonts/fontawesome-webfont' . $prefix . '.ttf?v=4.7.0") format("truetype"),
				url("' . $url . '/fonts/fontawesome-webfont' . $prefix . '.svg?v=4.7.0#fontawesomeregular") format("svg");
				font-weight: normal;
				font-style: normal;';

			if ( 'disable' !== $font_display ) {
				$output .= 'font-display:' . $font_display . ';';
			}

			$output .= '}';
		}

		return $output;
	}

	/**
	 * Get custom font url.
	 *
	 * @since 1.0.0
	 *
	 * @param array $font Font data.
	 *
	 * @return string
	 */
	public function get_custom_font_url( $font ) {
		$url = '';

		if ( isset( $font['id'] ) && $font['id'] ) {
			$url = wp_get_attachment_url( $font['id'] );
		} elseif ( is_array( $font ) ) {
			$url = $font['url'];
		}

		return woodmart_remove_https( $url );
	}

	/**
	 * Get theme settings css.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_theme_settings_css() {
		$widgets_scroll = woodmart_get_opt( 'widgets_scroll' );
		$widgets_height = woodmart_get_opt( 'widget_heights' );

		// Quick view
		$quick_view_width = woodmart_get_opt( 'quickview_width' );

		// Shop popup
		$shop_popup_width = woodmart_get_opt( 'popup_width' );

		// Header banner
		$header_banner_height        = woodmart_get_opt( 'header_banner_height' );
		$header_banner_height_mobile = woodmart_get_opt( 'header_banner_mobile_height' );

		$custom_product_background = get_post_meta( get_the_ID(), '_woodmart_product-background', true );

		$site_custom_width     = woodmart_get_opt( 'site_custom_width' );
		$predefined_site_width = woodmart_get_opt( 'site_width' );

		$site_width = '';

		if ( $predefined_site_width == 'full-width' ) {
			$site_width = 1222;
		} elseif ( $predefined_site_width == 'boxed-2' ) {
			$site_width = 1160;
		} elseif ( $predefined_site_width == 'wide' ) {
			$site_width = 1600;
		} elseif ( $predefined_site_width == 'custom' ) {
			$site_width = $site_custom_width;
		}

		ob_start();
		// phpcs:disable
		?>
		
		<?php if ( $site_width ): ?>
			/* Site width */
			
			/* Header Boxed */
			@media (min-width: 1025px) {
			
			.whb-boxed:not(.whb-sticked):not(.whb-full-width) .whb-main-header {
			max-width: <?php echo esc_html( $site_width - 30 ); ?>px;
			}
			}
			
			.container {
			max-width: <?php echo esc_html( $site_width ); ?>px;
			}
			
			@media (min-width: <?php echo esc_html( $site_width + 70 ); ?>px) {
			
			[data-vc-full-width] {
			<?php if ( is_rtl() ): ?>
				left: calc((100vw - <?php echo esc_html( $site_width ); ?>px) / 2);
			<?php else: ?>
				left: calc((-100vw - -<?php echo esc_html( $site_width ); ?>px) / 2);
			<?php endif; ?>
			}
			
			[data-vc-full-width]:not([data-vc-stretch-content]) {
			padding-left: calc((100vw - <?php echo esc_html( $site_width ); ?>px) / 2);
			padding-right: calc((100vw - <?php echo esc_html( $site_width ); ?>px) / 2);
			}
			
			.platform-Windows [data-vc-full-width] {
			<?php if ( is_rtl() ): ?>
				left: calc((100vw - <?php echo esc_html( $site_width + 17 ); ?>px) / 2);
			<?php else: ?>
				left: calc((-100vw - -<?php echo esc_html( $site_width + 17 ); ?>px) / 2);
			<?php endif; ?>
			}
			
			.platform-Windows [data-vc-full-width]:not([data-vc-stretch-content]) {
			padding-left: calc((100vw - <?php echo esc_html( $site_width + 17 ); ?>px) / 2);
			padding-right: calc((100vw - <?php echo esc_html( $site_width + 17 ); ?>px) / 2);
			}
			
			.browser-Edge [data-vc-full-width] {
			<?php if ( is_rtl() ): ?>
				left: calc((100vw - <?php echo esc_html( $site_width + 12 ); ?>px) / 2);
			<?php else: ?>
				left: calc((-100vw - -<?php echo esc_html( $site_width + 12 ); ?>px) / 2);
			<?php endif; ?>
			}
			
			.browser-Edge [data-vc-full-width]:not([data-vc-stretch-content]) {
			padding-left: calc((100vw - <?php echo esc_html( $site_width + 12 ); ?>px) / 2);
			padding-right: calc((100vw - <?php echo esc_html( $site_width + 12 ); ?>px) / 2);
			}
			}
		
		
		<?php endif; ?>
		
		/* Quick view */
		.popup-quick-view {
		max-width: <?php echo esc_html( $quick_view_width ); ?>px;
		}
		
		/* Shop popup */
		.woodmart-promo-popup {
		max-width: <?php echo esc_html( $shop_popup_width ); ?>px;
		}
		
		/* Header Banner */
		.header-banner {
		height: <?php echo esc_html( $header_banner_height ); ?>px;
		}
		
		.header-banner-display .website-wrapper {
		margin-top:<?php echo esc_html( $header_banner_height ); ?>px;
		}
		
		/* Tablet */
		@media (max-width: 1024px) {
		
		/* header Banner */
		
		.header-banner {
		height: <?php echo esc_html( $header_banner_height_mobile ); ?>px;
		}
		
		.header-banner-display .website-wrapper {
		margin-top:<?php echo esc_html( $header_banner_height_mobile ); ?>px;
		}
		
		}
		
		<?php if( $widgets_scroll ): ?>
			.woodmart-woocommerce-layered-nav .woodmart-scroll-content {
			max-height: <?php echo esc_attr( $widgets_height ); ?>px;
			}
		<?php endif; ?>
		
		<?php if ( !empty( $custom_product_background ) ): ?>
			.single-product .main-page-wrapper{
			background-color: <?php echo esc_html( $custom_product_background ); ?> !important;
			}
		<?php endif ?>

		<?php

		return ob_get_clean();
	}
}
